/* -LICENSE-START-
 ** Copyright (c) 2015 Blackmagic Design
 **
 ** Permission is hereby granted, free of charge, to any person or organization
 ** obtaining a copy of the software and accompanying documentation covered by
 ** this license (the "Software") to use, reproduce, display, distribute,
 ** execute, and transmit the Software, and to prepare derivative works of the
 ** Software, and to permit third-parties to whom the Software is furnished to
 ** do so, all subject to the following:
 **
 ** The copyright notices in the Software and this entire statement, including
 ** the above license grant, this restriction and the following disclaimer,
 ** must be included in all copies of the Software, in whole or in part, and
 ** all derivative works of the Software, unless such copies or derivative
 ** works are solely in the form of machine-executable object code generated by
 ** a source language processor.
 **
 ** THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 ** IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 ** FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
 ** SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
 ** FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
 ** ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 ** DEALINGS IN THE SOFTWARE.
 ** -LICENSE-END-
 */

#include "ControllerWidget.h"

#include "CommonGui.h"
#include "ColourPalette.h"
#include "CommonWidgets.h"

#include <QTime>
#include <QTimer>

#define PSS(x) ColourPalette::processStyleSheet(x)

static const int kMegabit = 1000000;

ControlsWidget::ControlsWidget() : m_frameRate(0)
{
	setObjectName("ControlsWidget");
	setFixedSize(598, 200);
	setStyleSheet(PSS(
					  "QWidget { background: kColour3; }"
					  "QLabel { margin: 0; spacing: 0;}"
					  "QWidget#ControlsWidget { border: 1px solid kColour2; border-radius: 4px; padding-top: 18px; padding-left: 20px; padding-right: 20px; padding-bottom: 20px;}"
					  ));

	QVBoxLayout* layout = new TightVBoxLayout;

	QHBoxLayout* layoutR1 = new TightHBoxLayout;
	QVBoxLayout* vertLayout = new TightVBoxLayout;

	QLabel* formatLabel = new QLabel("VIDEO FORMAT");
	formatLabel->setFont(CommonGui::font(CommonGui::kIcemanSemibold));
	formatLabel->setStyleSheet(PSS("QLabel { color: kColour5; padding-top: -3px;}"));
	formatLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
	formatLabel->setIndent(0);
	formatLabel->setFixedSize(138, 22);
	vertLayout->addWidget(formatLabel);

	m_modeLabel = new QLabel("Unknown");
    m_modeLabel->setFixedWidth(219);
	m_modeLabel->setStyleSheet(PSS("QLabel { color: kColour4; padding-top: -3px;}"));
	m_modeLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
	m_modeLabel->setIndent(0);
	m_modeLabel->setFont(CommonGui::font(CommonGui::kGalactusLight));
	vertLayout->addWidget(m_modeLabel);

	vertLayout->addStretch();
	layoutR1->addLayout(vertLayout);

	vertLayout = new TightVBoxLayout;

	formatLabel = new QLabel("TARGET BITRATE");
	formatLabel->setStyleSheet(PSS("QLabel { color: kColour5; padding-top: -3px;}"));
	formatLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
	formatLabel->setIndent(0);
	formatLabel->setFixedSize(316, 25);
	formatLabel->setFont(CommonGui::font(CommonGui::kIcemanSemibold));
	vertLayout->addWidget(formatLabel);

	m_speedSlider = new QSlider(Qt::Horizontal);
    m_speedSlider->setFixedSize(220,20);
	m_speedSlider->setStyleSheet(PSS(
                                     "QSlider::groove:horizontal { border: none; height: 23px; background: none; margin-top: -9px; margin-bottom: -10px; }"
									 "QSlider::handle:horizontal { image: url(:/slider_handle); }"
									 "QSlider::handle:horizontal:hover { image: url(:/slider_handle_focus); }"
									 "QSlider::handle:horizontal::disabled { image: url(:/slider_handle); }"
									 "QSlider::handle:horizontal:hover::disabled { image: url(:/slider_handle); }"
									 "QSlider::sub-page:horizontal { background: kColour2; }"
									 "QSlider::sub-page:horizontal::disabled { background: kColour2; }"
									 "QSlider::add-page:horizontal { background: kColour2; }"
									 "QSlider::sub-page:horizontal:disabled { background: kColour2; }"
									 "QSlider::add-page:horizontal:disabled { background: kColour2; }"
									 ));
	m_speedSlider->setMinimum(2.5 * kMegabit / kSpeedStepSize);
	m_speedSlider->setMaximum(50 * kMegabit / kSpeedStepSize);
	m_speedSlider->setValue(20 * kMegabit / kSpeedStepSize);
	m_speedSlider->setSingleStep(1 * kMegabit / kSpeedStepSize);
	connect(m_speedSlider, SIGNAL(valueChanged(int)), this, SLOT(onSpeedChanged(int)));
	vertLayout->addWidget(m_speedSlider);

	vertLayout->addStretch();
	layoutR1->addLayout(vertLayout);

	layoutR1->addSpacing(14);

	m_speedLabel = new QLabel("20Mb/s");
	m_speedLabel->setAlignment(Qt::AlignTop | Qt::AlignLeft);
	m_speedLabel->setIndent(0);
	m_speedLabel->setFont(CommonGui::font(CommonGui::kGalactusLight));
	m_speedLabel->setStyleSheet(PSS("QLabel { color: kColour4; padding-top: 19px;}"));
	layoutR1->addWidget(m_speedLabel, 0, Qt::AlignTop | Qt::AlignRight);

	layout->addLayout(layoutR1);

	QWidget* line = new QWidget();
	line->setFixedSize(558,1);
	line->setStyleSheet(PSS("QWidget { background: kColour2; }"));
	layout->addWidget(line, 0, Qt::AlignHCenter);

	QHBoxLayout* layoutR3 = new TightHBoxLayout;
	layoutR3->addSpacing(18);

	m_recordButton = new FlatImageButton(":/Record_default");
	layoutR3->addWidget(m_recordButton, 0, Qt::AlignBottom);

	layoutR3->addSpacing(40);

	QVBoxLayout* tcLayout = new TightVBoxLayout;

	tcLayout->addSpacing(27);

	m_tcLabel = new TimeCodeLabel();
	tcLayout->QLayout::addWidget(m_tcLabel);

	layoutR3->addLayout(tcLayout);

	layout->addLayout(layoutR3);

	setLayout(layout);

	m_updateTimer = new QTimer(this);
	connect(m_updateTimer, SIGNAL(timeout()), this, SLOT(onUpdateTime()));
}

void ControlsWidget::onSpeedChanged(int value)
{
	int bitrate = value * kSpeedStepSize;
	m_speedLabel->setText(QString("%1Mb/s").arg((double)bitrate / kMegabit));
	emit speedChanged(bitrate);
}

void ControlsWidget::onRecordingStarted(QString displayMode, uint32_t frameRate)
{
	m_frameRate = frameRate;
    m_modeLabel->setText(!displayMode.isEmpty() ? displayMode : "Unknown");

	m_speedSlider->setEnabled(false);
	m_recordButton->setImage(":/Record_stop");

	m_recordTime.start();
	m_updateTimer->start(1000 / frameRate);
}

void ControlsWidget::onRecordingStopped()
{
	m_updateTimer->stop();
	m_recordedDurationMs = m_recordTime.elapsed();

	m_recordButton->setImage(":/Record_default");
	m_speedSlider->setEnabled(true);

	m_frameRate = 0;
}

void ControlsWidget::onUpdateTime()
{
	updateTimeLabel(m_recordTime.elapsed());
}

void ControlsWidget::updateTimeLabel(int ms)
{
	m_tcLabel->setTime(ms, m_frameRate);
}

TimeCodeLabel::TimeCodeLabel()
{
	setStyleSheet(PSS("QLabel { color: kColour4;}"));
	setFixedHeight(70 - (29 - 20));

	QHBoxLayout* hLayout = new TightHBoxLayout;

	for(int i = 0; i < m_labels.size(); ++i)
	{
		if (i && !(i & 1))
		{
			QLabel* separator = new QLabel(":");
			separator->setIndent(0);
			separator->setAlignment(Qt::AlignTop | Qt::AlignCenter);
			separator->setFont(CommonGui::font(CommonGui::kVoltronLight));
			separator->setFixedWidth(5);
			hLayout->addWidget(separator);
		}

		m_labels[i] = new QLabel("0");
		m_labels[i]->setIndent(0);
		m_labels[i]->setAlignment(Qt::AlignTop | Qt::AlignCenter);
		m_labels[i]->setFont(CommonGui::font(CommonGui::kVoltronLight));
		m_labels[i]->setFixedWidth(40);
		hLayout->addWidget(m_labels[i]);
	}

	setTime(0, 24);
	setLayout(hLayout);
}

void TimeCodeLabel::setTime(int ms, uint32_t frameRate)
{
	QTime time = QTime::fromMSecsSinceStartOfDay(ms);
	int frame = time.msec() / (1000 / frameRate);
	m_labels[0]->setText(QString("%1").arg(time.hour() / 10));
	m_labels[1]->setText(QString("%1").arg(time.hour() % 10));
	m_labels[2]->setText(QString("%1").arg(time.minute() / 10));
	m_labels[3]->setText(QString("%1").arg(time.minute() % 10));
	m_labels[4]->setText(QString("%1").arg(time.second() / 10));
	m_labels[5]->setText(QString("%1").arg(time.second() % 10));
	m_labels[6]->setText(QString("%1").arg(frame / 10));
	m_labels[7]->setText(QString("%1").arg(frame % 10));
}
